// Game_Music_Emu 0.4.0. http://www.slack.net/~ant/

#include "Vgm_Emu.h"

#include <math.h>
#include <string.h>
#include "blargg_endian.h"

/* Copyright (C) 2003-2006 Shay Green. This module is free software; you
can redistribute it and/or modify it under the terms of the GNU Lesser
General Public License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version. This
module is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for
more details. You should have received a copy of the GNU Lesser General
Public License along with this module; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

#include "blargg_source.h"

static Music_Emu* new_vgm_emu() { return BLARGG_NEW Vgm_Emu; }

gme_type_t_ const gme_vgm_type [1] = { &new_vgm_emu, &Vgm_Emu::read_info };

double const gain = 3.0; // FM emulators are internally quieter to avoid 16-bit overflow
double const rolloff = 0.990;
double const oversample_factor = 1.5;

Vgm_Emu::Vgm_Emu( bool os, double tempo )
{
	oversample = os;
	pos = NULL;
	data = NULL;
	uses_fm = false;
	vgm_rate = (long) (44100 * tempo + 0.5);
	
	static equalizer_t const eq = { -14.0, 80 };
	set_equalizer( eq );
	psg.volume( 1.0 );
	blargg_verify_byte_order();
}

Vgm_Emu::~Vgm_Emu() { unload(); }

void Vgm_Emu::unload()
{
	data = NULL;
	pos = NULL;
	mem.clear();
	Music_Emu::unload();
}

blargg_err_t Vgm_Emu::set_sample_rate_( long sample_rate )
{
	RETURN_ERR( blip_buf.set_sample_rate( sample_rate, 1000 / 30 ) );
	return Classic_Emu::set_sample_rate_( sample_rate );
}

void Vgm_Emu::update_eq( blip_eq_t const& eq )
{
	psg.treble_eq( eq );
	dac_synth.treble_eq( eq );
}

void Vgm_Emu::set_voice( int i, Blip_Buffer* c, Blip_Buffer* l, Blip_Buffer* r )
{
	if ( i < psg.osc_count )
		psg.osc_output( i, c, l, r );
}

const char** Vgm_Emu::voice_names() const
{
	static const char* fm_names [] = {
		"FM 1", "FM 2", "FM 3", "FM 4", "FM 5", "FM 6", "PCM", "PSG"
	};
	if ( uses_fm )
		return fm_names;
	
	static const char* psg_names [] = { "Square 1", "Square 2", "Square 3", "Noise" };
	return psg_names;
}

void Vgm_Emu::mute_voices( int mask )
{
	Classic_Emu::mute_voices( mask );
	dac_synth.output( &blip_buf );
	if ( uses_fm )
	{
		psg.output( (mask & 0x80) ? 0 : &blip_buf );
		if ( ym2612.enabled() )
		{
			dac_synth.volume( (mask & 0x40) ? 0.0 : 0.1115 / 256 * gain );
			ym2612.mute_voices( mask );
		}
		
		if ( ym2413.enabled() )
		{
			int m = mask & 0x3F;
			if ( mask & 0x20 )
				m |= 0x01E0; // channels 5-8
			if ( mask & 0x40 )
				m |= 0x3E00;
			ym2413.mute_voices( m );
		}
	}
}

blargg_err_t Vgm_Emu::load_( const header_t& h, void const* new_data, long new_size )
{
	header_ = h;
	
	// compatibility
	if ( 0 != memcmp( header_.tag, "Vgm ", 4 ) )
		return "Not a VGM file";
	check( get_le32( header_.version ) <= 0x150 );
	
	// psg rate
	long psg_rate = get_le32( header_.psg_rate );
	if ( !psg_rate )
		psg_rate = 3579545;
	blip_time_factor = (long) floor( (double) (1L << blip_time_bits) / vgm_rate * psg_rate + 0.5 );
	blip_buf.clock_rate( psg_rate );
	
	data = (byte*) new_data;
	data_end = data + new_size;
	
	// get loop
	loop_begin = data_end;
	if ( get_le32( header_.loop_offset ) )
		loop_begin = &data [get_le32( header_.loop_offset ) + offsetof (header_t,loop_offset) - 0x40];
	
	set_voice_count( psg.osc_count );
	set_track_count( 1 );
	
	RETURN_ERR( setup_fm() );
	
	// do after FM in case output buffer is changed
	RETURN_ERR( Classic_Emu::setup_buffer( psg_rate ) );
	
	remute_voices(); // assigns PSG and DAC to proper Blip_Buffers
	
	return 0;
}

blargg_err_t Vgm_Emu::setup_fm()
{
	long ym2612_rate = get_le32( header_.ym2612_rate );
	long ym2413_rate = get_le32( header_.ym2413_rate );
	if ( ym2413_rate && get_le32( header_.version ) < 0x110 )
		update_fm_rates( &ym2413_rate, &ym2612_rate );
	
	uses_fm = false;
	
	double fm_rate = blip_buf.sample_rate() * oversample_factor;
	
	if ( ym2612_rate )
	{
		uses_fm = true;
		if ( !oversample )
			fm_rate = ym2612_rate / 144.0;
		Dual_Resampler::setup( fm_rate / blip_buf.sample_rate(), rolloff, gain );
		RETURN_ERR( ym2612.set_rate( fm_rate, ym2612_rate ) );
		ym2612.enable( true );
		set_voice_count( 8 );
	}
	
	if ( !uses_fm && ym2413_rate )
	{
		uses_fm = true;
		if ( !oversample )
			fm_rate = ym2413_rate / 72.0;
		Dual_Resampler::setup( fm_rate / blip_buf.sample_rate(), rolloff, gain );
		int result = ym2413.set_rate( fm_rate, ym2413_rate );
		if ( result == 2 )
			return "YM2413 FM sound isn't supported";
		CHECK_ALLOC( !result );
		ym2413.enable( true );
		set_voice_count( 8 );
	}
	
	if ( uses_fm )
	{
		fm_time_factor = 2 + (long) floor( fm_rate * (1L << fm_time_bits) / vgm_rate + 0.5 );
		RETURN_ERR( Dual_Resampler::resize( blip_buf.length() * blip_buf.sample_rate() / 1000 ) );
		psg.volume( 0.135 * gain );
	}
	else
	{
		ym2612.enable( false );
		ym2413.enable( false );
		psg.volume( 1.0 );
	}
	
	return 0;
}

blargg_err_t Vgm_Emu::load( Data_Reader& reader )
{
	header_t h;
	RETURN_ERR( reader.read( &h, sizeof h ) );
	return load( h, reader );
}

blargg_err_t Vgm_Emu::load( const header_t& h, Data_Reader& reader )
{
	unload();
	
	// allocate and read data
	long data_size = reader.remain();
	int const padding = 8;
	RETURN_ERR( mem.resize( data_size + padding ) );
	blargg_err_t err = reader.read( mem.begin(), data_size );
	if ( err )
	{
		unload();
		return err;
	}
	memset( &mem [data_size], 0x66, padding ); // pad with end command
	
	return load_( h, mem.begin(), data_size );
}

void Vgm_Emu::start_track_( int track )
{
	require( data ); // file must have been loaded
	
	Classic_Emu::start_track_( track );
	psg.reset( get_le16( header_.noise_feedback ), header_.noise_width );
	
	dac_disabled = -1;
	pcm_data = data;
	pcm_pos = data;
	dac_amp = -1;
	vgm_time = 0;
	pos = data;
	if ( get_le32( header_.version ) >= 0x150 )
	{
		long data_offset = get_le32( header_.data_offset );
		check( data_offset );
		if ( data_offset )
			pos += data_offset + offsetof (header_t,data_offset) - 0x40;
	}
	
	if ( uses_fm )
	{
		if ( ym2413.enabled() )
			ym2413.reset();
		
		if ( ym2612.enabled() )
			ym2612.reset();
		
		fm_time_offset = 0;
		blip_buf.clear();
		Dual_Resampler::clear();
	}
}

long Vgm_Emu::run( int msec, bool* added_stereo )
{
	blip_time_t psg_end = run_commands( msec * vgm_rate / 1000 );
	*added_stereo = psg.end_frame( psg_end );
	return psg_end;
}

void Vgm_Emu::play_( long count, sample_t* out )
{
	require( pos ); // track must have been started
	
	if ( uses_fm )
		Dual_Resampler::play( count, out, blip_buf );
	else
		Classic_Emu::play_( count, out );
}

// Track info

static byte const* skip_gd3_str( byte const* in, byte const* end )
{
	while ( end - in >= 2 )
	{
		in += 2;
		if ( !(in [-2] | in [-1]) )
			break;
	}
	return in;
}

static byte const* get_gd3_str( byte const* in, byte const* end, track_info_t* out, char* field )
{
	byte const* mid = skip_gd3_str( in, end );
	int len = (mid - in) / 2 - 1;
	if ( len > 0 )
	{
		len = min( len, (int) out->max_field );
		field [len] = 0;
		for ( int i = 0; i < len; i++ )
			field [i] = (in [i * 2 + 1] ? '?' : in [i * 2]); // TODO: convert to utf-8
	}
	return mid;
}

static byte const* get_gd3_pair( byte const* in, byte const* end,
		track_info_t* out, char* field )
{
	return skip_gd3_str( get_gd3_str( in, end, out, field ), end );
}

static void parse_gd3( byte const* in, byte const* end, track_info_t* out )
{
	in = get_gd3_pair( in, end, out, out->song );
	in = get_gd3_pair( in, end, out, out->game );
	in = get_gd3_pair( in, end, out, out->system );
	in = get_gd3_pair( in, end, out, out->author );
	in = get_gd3_str ( in, end, out, out->copyright );
	in = get_gd3_pair( in, end, out, out->dumper );
	in = get_gd3_str ( in, end, out, out->comment );
}

int const gd3_header_size = 12;

static long check_gd3_header( byte const* h, long remain )
{
	if ( remain < gd3_header_size ) return 0;
	if ( memcmp( h, "Gd3 ", 4 ) ) return 0;
	if ( get_le32( h + 4 ) >= 0x200 ) return 0;
	
	long gd3_size = get_le32( h + 8 );
	if ( gd3_size > remain - gd3_header_size ) return 0;
	
	return gd3_size;
}

byte const* Vgm_Emu::gd3_data( int* size ) const
{
	if ( size )
		*size = 0;
	
	long gd3_offset = get_le32( header_.gd3_offset ) - 0x2C;
	if ( gd3_offset < 0 )
		return 0;
	
	byte const* gd3 = data + gd3_offset;
	long gd3_size = check_gd3_header( gd3, data_end - gd3 );
	if ( !gd3_size )
		return 0;
	
	if ( size )
		*size = gd3_size + gd3_header_size;
	
	return gd3;
}

static void get_vgm_length( Vgm_Emu::header_t const& h, track_info_t* out )
{
	long length = get_le32( h.track_duration ) * 10 / 441;
	if ( length > 0 )
	{
		long loop = get_le32( h.loop_duration );
		if ( loop > 0 && get_le32( h.loop_offset ) )
		{
			out->loop_length = loop * 10 / 441;
			out->intro_length = length - out->loop_length;
		}
		else
		{
			out->length = length; // 1000 / 44100 (VGM files used 44100 as timebase)
			out->intro_length = length; // make it clear that track is no longer than length
			out->loop_length = 0;
		}
	}
}

blargg_err_t Vgm_Emu::track_info( track_info_t* out, int track ) const
{
	if ( Music_Emu::track_info( out, track ) ) { }
	get_vgm_length( header_, out );
	
	int size;
	byte const* gd3 = gd3_data( &size );
	if ( gd3 )
		parse_gd3( gd3 + gd3_header_size, gd3 + size, out );
	
	return out->error();
}

blargg_err_t Vgm_Emu::read_info( Data_Reader& in, track_info_t* out, int track )
{
	header_t h;
	RETURN_ERR( in.read( &h, sizeof h ) );
	out->init();
	get_vgm_length( h, out );
	
	long gd3_offset = get_le32( h.gd3_offset ) - 0x2C;
	long remain = in.remain() - gd3_offset;
	byte gd3_h [gd3_header_size];
	if ( gd3_offset > 0 || remain >= gd3_header_size )
	{
		RETURN_ERR( in.skip( gd3_offset ) );
		RETURN_ERR( in.read( gd3_h, sizeof gd3_h ) );
		long gd3_size = check_gd3_header( gd3_h, remain );
		if ( gd3_size )
		{
			blargg_vector<byte> gd3;
			RETURN_ERR( gd3.resize( gd3_size ) );
			RETURN_ERR( in.read( gd3.begin(), gd3.size() ) );
			parse_gd3( gd3.begin(), gd3.end(), out );
		}
	}
	
	return out->error();
}
